import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Calendar, User, BookOpen, Star, TrendingUp, Award, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyLessons.css'

const MyLessons = () => {
  const [lessons, setLessons] = useState({ active: [], past: [] })
  const [activeTab, setActiveTab] = useState('active')

  useEffect(() => {
    loadLessons()
    const interval = setInterval(loadLessons, 500)
    return () => clearInterval(interval)
  }, [])

  const loadLessons = () => {
    const saved = localStorage.getItem('tutoringLessons')
    if (saved) {
      try {
        setLessons(JSON.parse(saved))
      } catch (e) {
        setLessons({ active: [], past: [] })
      }
    } else {
      const defaultLessons = {
        active: [
          {
            id: 1,
            subject: 'Математика',
            tutor: 'Анна Петрова',
            date: '2025-03-28',
            time: '15:00',
            status: 'active'
          },
          {
            id: 2,
            subject: 'Физика',
            tutor: 'Иван Сидоров',
            date: '2025-03-29',
            time: '16:30',
            status: 'active'
          }
        ],
        past: [
          {
            id: 3,
            subject: 'Математика',
            tutor: 'Анна Петрова',
            date: '2025-03-20',
            time: '15:00',
            efficiency: 5
          },
          {
            id: 4,
            subject: 'Английский',
            tutor: 'Мария Козлова',
            date: '2025-03-18',
            time: '14:00',
            efficiency: 4
          },
          {
            id: 5,
            subject: 'Русский',
            tutor: 'Елена Смирнова',
            date: '2025-03-15',
            time: '13:00',
            efficiency: 5
          }
        ]
      }
      setLessons(defaultLessons)
      localStorage.setItem('tutoringLessons', JSON.stringify(defaultLessons))
    }
  }

  const stats = [
    { 
      icon: BookOpen, 
      value: lessons.past.length, 
      label: 'Пройдено уроков',
      color: '#F97316'
    },
    { 
      icon: User, 
      value: new Set(lessons.past.map(l => l.tutor)).size, 
      label: 'Репетиторов',
      color: '#FB923C'
    },
    { 
      icon: Star, 
      value: lessons.past.length > 0 
        ? Math.round(lessons.past.reduce((sum, l) => sum + (l.efficiency || 0), 0) / lessons.past.length * 10) / 10
        : 0, 
      label: 'Средняя оценка',
      color: '#FDBA74'
    }
  ]

  const renderStars = (rating) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        size={18}
        fill={i < rating ? '#F59E0B' : 'none'}
        color={i < rating ? '#F59E0B' : '#D1D5DB'}
      />
    ))
  }

  return (
    <div className="my-lessons">
      <section className="page-intro">
        <motion.div
          className="intro-content"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="intro-icon-wrapper">
            <BookOpen className="intro-main-icon" size={64} color="#F97316" />
          </div>
          <h1 className="intro-title">Мои уроки</h1>
          <p className="intro-description">
            Управляйте своими онлайн-уроками с репетиторами. Отслеживайте прогресс и оценивайте эффективность занятий
          </p>
        </motion.div>
      </section>

      <div className="container">
        <ImageSlider />

        <div className="content-layout">
          <div className="sidebar-stats">
            <h2 className="sidebar-title">Ваша статистика</h2>
            {stats.map((stat, index) => (
              <motion.div
                key={stat.label}
                className="stat-card-vertical"
                initial={{ opacity: 0, x: -20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ x: 5, boxShadow: '0 8px 24px rgba(249, 115, 22, 0.2)' }}
              >
                <div className="stat-icon-wrapper" style={{ background: `${stat.color}20` }}>
                  <stat.icon className="stat-icon" size={28} color={stat.color} />
                </div>
                <div className="stat-content-vertical">
                  <div className="stat-value">{stat.value}</div>
                  <div className="stat-label">{stat.label}</div>
                </div>
              </motion.div>
            ))}
          </div>

          <div className="main-content">
            <div className="tabs">
              <button
                className={`tab ${activeTab === 'active' ? 'active' : ''}`}
                onClick={() => setActiveTab('active')}
              >
                Активные ({lessons.active.length})
              </button>
              <button
                className={`tab ${activeTab === 'past' ? 'active' : ''}`}
                onClick={() => setActiveTab('past')}
              >
                История ({lessons.past.length})
              </button>
            </div>

            <div className="lessons-section">
              <div className="lessons-grid">
                {activeTab === 'active' ? (
                  lessons.active.length > 0 ? (
                    lessons.active.map((lesson) => (
                      <motion.div
                        key={lesson.id}
                        className="lesson-card"
                        initial={{ opacity: 0, scale: 0.9 }}
                        animate={{ opacity: 1, scale: 1 }}
                        whileHover={{ y: -8, boxShadow: '0 12px 32px rgba(249, 115, 22, 0.2)' }}
                      >
                        <div className="lesson-header">
                          <div className="lesson-icon-wrapper">
                            <BookOpen className="lesson-icon" size={36} color="#F97316" />
                          </div>
                          <div className="lesson-info">
                            <h3 className="lesson-subject">{lesson.subject}</h3>
                            <p className="lesson-tutor">{lesson.tutor}</p>
                          </div>
                        </div>
                        <div className="lesson-details">
                          <div className="lesson-detail">
                            <Calendar className="detail-icon" size={20} color="#78716C" />
                            <span>
                              {new Date(lesson.date).toLocaleDateString('ru-RU', {
                                day: 'numeric',
                                month: 'long',
                                year: 'numeric'
                              })}
                            </span>
                          </div>
                          <div className="lesson-detail">
                            <Clock className="detail-icon" size={20} color="#78716C" />
                            <span>{lesson.time}</span>
                          </div>
                        </div>
                        <div className="lesson-status active-badge">
                          Запланирован
                        </div>
                      </motion.div>
                    ))
                  ) : (
                    <div className="empty-state">
                      <BookOpen className="empty-icon" size={72} color="#D1D5DB" />
                      <p>У вас нет активных уроков</p>
                    </div>
                  )
                ) : (
                  lessons.past.length > 0 ? (
                    lessons.past.map((lesson) => (
                      <motion.div
                        key={lesson.id}
                        className="lesson-card"
                        initial={{ opacity: 0, scale: 0.9 }}
                        animate={{ opacity: 1, scale: 1 }}
                        whileHover={{ y: -8, boxShadow: '0 12px 32px rgba(249, 115, 22, 0.2)' }}
                      >
                        <div className="lesson-header">
                          <div className="lesson-icon-wrapper">
                            <BookOpen className="lesson-icon" size={36} color="#F97316" />
                          </div>
                          <div className="lesson-info">
                            <h3 className="lesson-subject">{lesson.subject}</h3>
                            <p className="lesson-tutor">{lesson.tutor}</p>
                          </div>
                        </div>
                        <div className="lesson-details">
                          <div className="lesson-detail">
                            <Calendar className="detail-icon" size={20} color="#78716C" />
                            <span>
                              {new Date(lesson.date).toLocaleDateString('ru-RU', {
                                day: 'numeric',
                                month: 'long',
                                year: 'numeric'
                              })}
                            </span>
                          </div>
                          <div className="lesson-detail">
                            <Clock className="detail-icon" size={20} color="#78716C" />
                            <span>{lesson.time}</span>
                          </div>
                        </div>
                        {lesson.efficiency && (
                          <div className="lesson-efficiency">
                            <div className="efficiency-label">
                              <TrendingUp className="efficiency-icon" size={18} color="#F97316" />
                              <span>Эффективность:</span>
                            </div>
                            <div className="stars-wrapper">
                              {renderStars(lesson.efficiency)}
                            </div>
                          </div>
                        )}
                      </motion.div>
                    ))
                  ) : (
                    <div className="empty-state">
                      <BookOpen className="empty-icon" size={72} color="#D1D5DB" />
                      <p>История уроков пуста</p>
                    </div>
                  )
                )}
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

export default MyLessons

